/*
 * Tigase Jabber/XMPP Server
 * Copyright (C) 2004-2016 "Tigase, Inc." <office@tigase.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. Look for COPYING file in the top folder.
 * If not, see http://www.gnu.org/licenses/.
 */

package tigase.licence.utils;

import tigase.xml.DomBuilderHandler;
import tigase.xml.Element;
import tigase.xml.SimpleParser;
import tigase.xml.SingletonFactory;

import javax.net.ssl.*;
import javax.xml.ws.http.HTTPException;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.*;
import java.security.cert.CertificateException;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

public class DataLoader {

	private static final Logger log = Logger.getLogger(DataLoader.class.getCanonicalName());

	private static final String KEYSTORE = "certs/keystore-letsencrypt";
//	private static final String KEYSTORE = "library/src/main/resources/keystore-letsencrypt";
//	private static final String KEYSTORE = "src/main/resources/keystore-letsencrypt";

	private static final String KEYSTORE_PASSWORD = "keystore";

	private static final int DEFAULT_REQUEST_TIMEOUT = 15;

	private SimpleParser parser = SingletonFactory.getParserInstance();
	private SSLSocketFactory sslSocketFactory = null;

	public static <T> void addRequestItems(StringBuffer rawData, String fieldVar, List<T> items) {
		rawData.append("<item><var>").append(fieldVar).append("</var>");
		rawData.append("<value>");
		if (items != null && !items.isEmpty()) {
			for (T hash : items) {
				rawData.append("<item>").append(hash).append("</item>");
			}
		}
		rawData.append("</value>").append("</item>");
	}

	private SSLSocketFactory getSslSocketFactory()
			throws KeyStoreException, IOException, NoSuchAlgorithmException, CertificateException,
			       UnrecoverableKeyException, KeyManagementException {
		if (sslSocketFactory == null) {
			KeyStore ks = KeyStore.getInstance("JKS");
			ks.load(new FileInputStream(KEYSTORE), KEYSTORE_PASSWORD.toCharArray());

			KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
			kmf.init(ks, KEYSTORE_PASSWORD.toCharArray());

			TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
			tmf.init(ks);

			SSLContext sslContext = SSLContext.getInstance("TLS");
			sslContext.init(kmf.getKeyManagers(), tmf.getTrustManagers(), null);

			sslSocketFactory = sslContext.getSocketFactory();
		}
		return sslSocketFactory;
	}

	public Element loadData(final String url, final String encodedData) throws TooManyRequestsHTTPException, GeneralSecurityException, IOException, HTTPException {
		String type = "application/xml";

		URL u = new URL(url);

		HttpURLConnection conn = (HttpURLConnection) u.openConnection();

		if ("https".equals(u.getProtocol())) {
			((HttpsURLConnection) conn).setSSLSocketFactory(getSslSocketFactory());
		}

		conn.setDoOutput(true);
		conn.setRequestMethod("POST");

		conn.setConnectTimeout(DEFAULT_REQUEST_TIMEOUT * 1000);
		conn.setReadTimeout(DEFAULT_REQUEST_TIMEOUT * 1000);

		conn.setRequestProperty("Accept", "*/*");

		conn.setRequestProperty("Content-Type", type);
		conn.setRequestProperty("Content-Length", String.valueOf(encodedData.length()));
		OutputStream os = conn.getOutputStream();

		if (log.isLoggable(Level.FINEST)) {
			log.log(Level.FINEST, "Sending data: {0} to url: {1}", new Object[]{encodedData, u});
		}

		os.write(encodedData.getBytes());
		os.flush();

		final int responseCode = conn.getResponseCode();
		final InputStream inputStream = responseCode / 100 == 2 ? conn.getInputStream() : conn.getErrorStream();
		if (responseCode == 429) {
			String retryStr = conn.getHeaderField("Retry-After");
			if (null != retryStr && !retryStr.isEmpty()) {
				int retryAfter = Integer.valueOf(retryStr);
				throw new TooManyRequestsHTTPException(responseCode, retryAfter);
			}
		} else if (responseCode / 100 != 2) {
			throw new HTTPException(responseCode);
		}


		BufferedReader in = new BufferedReader(new InputStreamReader(inputStream));
		String inputLine;

		StringBuilder sb = new StringBuilder();

		DomBuilderHandler handler = new DomBuilderHandler();
		while ((inputLine = in.readLine()) != null) {
			sb.append(inputLine);
			parser.parse(handler, inputLine.toCharArray(), 0, inputLine.length());
		}

		in.close();

		final Element result = handler.getParsedElements().poll();

		log.log(Level.FINEST, "Retrieved data from server, response: {0}, (parsed): {1}, sb: {2} ",
		        new Object[] {responseCode, result, sb});


		return result;
	}

}
