/*
 * Tigase Jabber/XMPP Server
 * Copyright (C) 2004-2012 "Artur Hefczyc" <artur.hefczyc@tigase.org>
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, version 3 of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. Look for COPYING file in the top folder.
 * If not, see http://www.gnu.org/licenses/.
 */
package tigase.licence;

import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import tigase.cluster.SessionManagerClustered;
import tigase.conf.Configurable;
import tigase.conf.ConfiguratorAbstract;
import tigase.server.MessageRouterConfig;
import tigase.server.XMPPServer;
import tigase.server.xmppsession.SessionManager;
import tigase.stats.StatisticsCollector;
import tigase.stats.collector.StatisticsData;
import tigase.sys.TigaseRuntime;
import tigase.vhosts.VHostManager;
import tigase.vhosts.VHostManagerIfc;
import tigase.xml.Element;
import tigase.xmpp.BareJID;
import tigase.xmpp.JID;

/**
 *
 * @author Wojciech Kapcia <wojciech.kapcia@tigase.org>
 */
public class LicenceCheckerUpdater {

	private static final Logger log = Logger.getLogger( "tigase.stats" );
	//~--- fields ---------------------------------------------------------------
	/** Holds configurator implementation which handles all configuration of
	 * Tigase */
	protected static ConfiguratorAbstract configurator;
	/** Holds {@link tigase.server.xmppsession.SessionManager} */
	protected static SessionManager sess;
	/** Holds {@link tigase.cluster.SessionManagerClustered} */
	protected static SessionManagerClustered sessclust;
	/** Holds {@link tigase.stats.StatisticsCollector} implementation which
	 * manages all statistics related functionality in Tigase */
	private static StatisticsCollector stats;
	/** Holds {@link tigase.vhosts.VHostManagerIfc} implementation which manages
	 * all virtual host related funcionality in Tigase */
	protected static VHostManagerIfc vhost;
	/**
	 * Constant
	 * <code>CLUSTER_MODE</code> sets the cluster mode to either
	 * <code>true</code> or
	 * <code>false</code>. By default cluster mode is set to
	 * <code>false</code>.
	 */
	public static final String CLUSTER_MODE = "cluster-mode";
	private static boolean clusterMode;

	//~--- constructors ---------------------------------------------------------
	/**
	 * Constructs default implementation of handling updates of statistics data on
	 * per-component basis providing most basic statistics data about system.
	 */
	static {
		init();
	}

	//~--- methods --------------------------------------------------------------
	public static StatisticsData updateData() {
		log.finest( "updating statistics data" );

		StatisticsData data = null;
		try {
			if ( configurator == null || sess == null || stats == null || vhost == null ){
				init();
			}
			data = new StatisticsData( getHostname() );

			data.addVHosts( getVHosts() );
			data.setUptime( getUptime() );
			data.setHeapUsed( getHeapUsed() );
			data.setHeapMax( getHeapMax() );
			data.setUsersOnline( getUserOnline() );
			data.setUsersActive( getUserActive() );
			data.setClusterNodesCount( getClusterNodesCount() );
			data.setMaxSessionsToday(getMaxSessionsToday());
			data.setMaxSessionsYesterday(getMaxSessionsYesterday());

			/*
			 * list all licenced component being used and whether they have valid licence
			 */
			data.setAdditionalData( getComponentInfo().toString() );


		} catch ( Exception ex ) {
			log.log( Level.SEVERE, "Problem creating StatisticsData", ex );
		}
		return data;
	}

	//~--- get methods ----------------------------------------------------------
	public String getMissingLicenseWarning() {
		//J-
		return "This installation contains Tigase ACS package, not an open source software.\n"
					 + "The Tigase ACS is only available under a commercial license.\n"
					 + "The full text of the commercial license agreement is available upon request.\n"
					 + "\n"
					 + "The Tigase ACS component is provided free of charge for testing and\n"
					 + "development purposes only. Any use of the component on production systems,\n"
					 + "either commercial or not-for-profit, requires the purchase of a license.\n"
					 + "\n"
					 + "If the Tigase ACS component is activated without a valid license, it will\n"
					 + "continue to work, including its full set of features, but it will send\n"
					 + "certain statistical information to Tigase's servers on a regular basis.\n"
					 + "If the Tigase ACS component cannot access our servers to send information,\n"
					 + "it will stop working. Once a valid license is installed, the Tigase ACS\n"
					 + "component will stop sending statistical information to Tigase's servers.\n"
					 + "\n"
					 + "By activating the Tigase ACS component without a valid license you agree\n"
					 + "and accept that the component will send certain statistical information\n"
					 + "(such as DNS domain names, vhost names, number of online users, number of\n"
					 + "cluster nodes, etc.) which may be considered confidential and proprietary\n"
					 + "by the user. You accept and confirm that such information, which may be\n"
					 + "considered confidential or proprietary, will be transferred to Tigase's\n"
					 + "servers and that you will not pursue any remedy at law as a result of the\n"
					 + "information transfer.\n"
					 + "If the Tigase ACS component is installed but not activated, no statistical\n"
					 + "information will be sent to Tigase's servers.";
		//J+
	}

	/**
	 * Returns Maxim Heap Memory usage, either OLD memory pool size as this is
	 * what is the most interesting to us or if this is not possible then total
	 * Heap size.
	 *
	 * @return maximum heap memory usage
	 */
	protected static long getHeapMax() {
		return TigaseRuntime.getTigaseRuntime().getHeapMemMax();
	}

	/**
	 * Returns Heap Memory usage, either OLD memory pool size as this is what is
	 * the most interesting to us or if this is not possible then total Heap size.
	 *
	 * @return heap memory usage
	 */
	protected static long getHeapUsed() {
		return TigaseRuntime.getTigaseRuntime().getHeapMemUsed();
	}

	/**
	 * Return default hostname of the node (i.e. machine hostname)
	 *
	 * @return hostname of the machine
	 */
	protected static BareJID getHostname() {
		BareJID hostname = null;
		try {
			String host = sess.getDefHostName().toString();
			hostname = BareJID.bareJIDInstance( host );
		} catch ( Exception ex ) {
			log.log( Level.SEVERE, "Error during obtaining hostname", ex );
		}
		return hostname;
	}

	/**
	 * Returns uptime of the instance.
	 *
	 * @return a  <code>long</code> value of the uptime in milisenconds
	 */
	protected static long getUptime() {
		return TigaseRuntime.getTigaseRuntime().getUptime();
	}

	public static StatisticsCollector getStats() {
		return stats;
	}

	/**
	 * Returns number of all online users connections on given node.
	 *
	 * @return a  <code>long</code> value representing all online users connection
	 *         on given node.
	 */
	protected static long getUserOnline() {
		long number;
		String sNum;
		sNum = (String) stats.getAllStats().getValue( "sess-man/Open user connections" );
		if ( sNum != null ){
			number = Long.valueOf( sNum );
		} else {
			System.out.println( "error getting number" );
			number = 0;
		}
		return number;
	}

	/**
	 * Returns number of all active users connections (i.e. which session wasn't
	 * idle more than 5 minutes) on given node.
	 *
	 * @return a  <code>long</code> value representing all online users connection
	 *         on given node.
	 */
	protected static long getUserActive() {
		long number;
		String sNum;
		sNum = (String) stats.getAllStats().getValue( "sess-man/Active user connections" );
		if ( sNum != null ){
			number = Long.valueOf( sNum );
		} else {
			System.out.println( "error getting number" );
			number = 0;
		}
		return number;
	}
	
	protected static long getMaxSessionsToday() {
		long number;
		String sNum;
		sNum = (String) stats.getAllStats().getValue( "sess-man/Maximum user sessions today" );
		if ( sNum != null ){
			number = Long.valueOf( sNum );
		} else {
			System.out.println( "error getting number" );
			number = 0;
		}
		return number;
	}
	
	protected static long getMaxSessionsYesterday() {
		long number;
		String sNum;
		sNum = (String) stats.getAllStats().getValue( "sess-man/Maximum user sessions yesterday" );
		if ( sNum != null ){
			number = Long.valueOf( sNum );
		} else {
			System.out.println( "error getting number" );
			number = 0;
		}
		return number;
	}

	/**
	 * Returns list of all Virtual Hosts on that particular Tigase instance.
	 *
	 * @return a  <code>BareJID </code> list containing all Virtual Hosts on that
	 *         particular instance
	 */
	protected static List<BareJID> getVHosts() {
		List<BareJID> list = new ArrayList<BareJID>();

		for ( JID jid : vhost.getAllVHosts() ) {
			list.add( jid.getBareJID() );
		}

		return list;
	}

	/**
	 * Returns Component Information about all system components
	 *
	 * @return an  <code>{@link tigase.xml.Element}</code> representation of all
	 *         system Components information
	 */
	protected static Element getComponentInfo() {

		Element result = new Element( "components" );

		try {
			Map<String, Object> mrProps = configurator.getProperties( "message-router" );
			List<String> compNames = new ArrayList<String>( 10 );
			compNames.addAll( Arrays.asList(
					(String[]) mrProps.get( MessageRouterConfig.MSG_RECEIVERS_NAMES_PROP_KEY ) ) );
			compNames.addAll( Arrays.asList(
					(String[]) mrProps.get( MessageRouterConfig.REGISTRATOR_NAMES_PROP_KEY ) ) );

			for ( String cmp : compNames ) {
				Configurable mr = configurator.getComponent( cmp );
				if ( mr != null ){
					result.addChild( mr.getComponentInfo().toElement() );
				}
			}
		} catch ( Exception ex ) {
			log.log( Level.SEVERE, "Problem obtaining Configurator", ex );
		}
		return result;
	}

	public static int getClusterNodesCount() {
		int size;
		if ( clusterMode && sessclust != null ){
			size = sessclust.getStrategy().getNodesConnected().size() + 1;
		} else {
			size = 0;
		}
		return size;
	}

	protected static void init() {
		configurator = XMPPServer.getConfigurator();
		if ( configurator != null ){
			sess = (SessionManager) configurator.getComponent( "sess-man" );
			stats = (StatisticsCollector) configurator.getComponent( "stats" );
			vhost = (VHostManager) configurator.getComponent( "vhost-man" );
			String clust = System.getProperty( CLUSTER_MODE );
			clusterMode = Boolean.valueOf( clust );
			if ( clusterMode ){
				sessclust = (SessionManagerClustered) configurator.getComponent( "sess-man" );
			}
		}
	}
}
