/*
 * Tigase Jabber/XMPP Server
 * Copyright (C) 2004-2016 "Tigase, Inc." <office@tigase.com>
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, version 3 of the License,
 * or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. Look for COPYING file in the top folder.
 * If not, see http://www.gnu.org/licenses/.
 */
package tigase.licence;

import tigase.db.RepositoryFactory;
import tigase.db.UserRepository;
import tigase.licence.utils.DataLoader;
import tigase.licence.utils.TooManyRequestsHTTPException;
import tigase.licence.utils.VHostsRetriever;
import tigase.xml.Element;
import tigase.xmpp.BareJID;
import tigase.xmpp.JID;

import javax.xml.ws.http.HTTPException;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;
import java.util.logging.Level;
import java.util.logging.Logger;

import static tigase.licence.LicenceChecker.INSTALLATION_ID_KEY;

/**
 * @author Wojtek
 */
public class InstallationIdRetriever {

	private static final Logger log = Logger.getLogger(InstallationIdRetriever.class.getCanonicalName());
	private static final String DEFAULT_INSTALLATION_ID_SERVER_URL = "https://license.tigase.net/register";
//	private static final String DEFAULT_INSTALLATION_ID_SERVER_URL = "http://127.0.0.1:8080/rest/licence/register";
//	private static final String DEFAULT_INSTALLATION_ID_SERVER_URL = "http://t2.tigase.org:8080/rest/licence/register";
	private static String installationId = null;
	private static InstallationIdRetriever instance = null;
	private static long lastCheck = 0;
	private DataLoader dl = new DataLoader();
	private final BareJID licenceUserName = BareJID.bareJIDInstanceNS("licence-library");
	private final Path INSTALLATION_ID_FILE = Paths.get("etc/installation-id");

	static InstallationIdRetriever getInstallationIdRetriever() {
		if (instance == null) {
			instance = new InstallationIdRetriever();
		}
		return instance;
	}

	public static void main(String[] args) {

//		Level lvl = Level.ALL;
//		ConsoleHandler handler = new ConsoleHandler();
//		handler.setLevel(lvl);
//		log.addHandler(handler);
//		log.setLevel(lvl);

		List<JID> j = new ArrayList<>();
		j.add(JID.jidInstanceNS("vhost1"));
		j.add(JID.jidInstanceNS("vhost2"));
		j.add(JID.jidInstanceNS("vhost3"));
		List<String> h = new ArrayList<>();
		h.add("hash1");
		h.add("hash2");
		h.add("hash3");
		final InstallationIdRetriever installationIdRetriever = InstallationIdRetriever.getInstallationIdRetriever();

		try {

//			final List<String> licenceDigests = installationIdRetriever.getLicenceFiles();
//			if (licenceDigests != null ) {
//				licenceDigests.forEach(System.out::println);
//			}

//			final String s1 = installationIdRetriever.retrieveInstallationId();
//			System.out.println(s1);
//
			Element element = null;

			try {
				element = installationIdRetriever.loadData(null, j, h);
			} catch (TooManyRequestsHTTPException e) {
				log.log(Level.WARNING, "Server returned error {0}: {1}; next retry in {2}s)",
				        new Object[]{e.getStatusCode(), e.getMessage(), e.getRetryAfter()});
			} catch (IOException | HTTPException e) {
				log.log(Level.WARNING, "Server returned error: {0}{1}",
				        new Object[]{e.getMessage(),
				                     (e instanceof HTTPException
				                        ? ", code: " + ((HTTPException) e).getStatusCode()
                                        : "")});
			}

			System.out.println("element: " + element);
			if (element != null) {
				final String installationID = element.getCDataStaticStr(
						new String[]{"command", "fields", "item", "value"});
				System.out.println("installationID: " + installationID);
			}

		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	private InstallationIdRetriever() {
		// init so the user DOES exist!
		try {
			String resource = System.getProperty(RepositoryFactory.GEN_USER_DB_URI_PROP_KEY);
			UserRepository userRepository = RepositoryFactory.getUserRepository(null, resource, null);

			if (!userRepository.userExists(licenceUserName)) {
				userRepository.addUser(licenceUserName);
			}
		} catch (Exception e) {
			log.log(Level.WARNING,
			        "There was a problem adding user {0} to the repository! Please make sure your database is configured correctly",
			        new Object[]{licenceUserName});
			if (log.isLoggable(Level.FINEST)) {
				log.log(Level.FINEST, "There was a problem adding user " + licenceUserName +
						" to the repository! Please make sure your database is configured correctly", e);
			}
		}
	}

	synchronized String getInstallationId() {
		// don't try to retrieve installation-id more often that 10min
		if ((System.currentTimeMillis() - lastCheck)/1000/60 < 10) {
			return installationId;
		} else {
			lastCheck = System.currentTimeMillis();
		}

		if (installationId == null) {

			// we should retry whole retrieval in case OTHER cluster node already got the id
			// from the server and stored it to the database (while the installation-id is not
			// based on the provided VHost list, the request with the same list will generate
			// 429 HTTP error code; other HTTP errors will also yield retry
			final double rate = 1.35;
			long retryAfter = 2;
			int retryLimit = 1;
			int retryCount = 0;
			do {
				try {
					installationId = retrieveInstallationId();

				} catch (TooManyRequestsHTTPException e) {
					retryLimit = 10;
					retryAfter = e.getRetryAfter() + 5;
					log.log(Level.WARNING, "Server returned error {0}: {1}; next retry in {2}s (retries left: {3})",
					        new Object[]{e.getStatusCode(), e.getMessage(), retryAfter, (retryLimit - retryCount)});
					try {
						Thread.sleep((retryAfter) * 1000);
					} catch (InterruptedException ex) {
						log.log(Level.FINE, "There was a problem delaying the subsequent request");
					}
				} catch (IOException | HTTPException e) {
					log.log(Level.WARNING, "Server returned error: {0}{1}; next retry in {2}s (retries left: {3})",
					        new Object[]{e.getMessage(), (e instanceof HTTPException
					                                      ? ", code: " + ((HTTPException) e).getStatusCode()
					                                      : ""), retryAfter, (retryLimit - retryCount)});
				}
				retryCount++;
			} while (installationId == null && retryCount < retryLimit);
		}
		return installationId;
	}

	/**
	 * Helper method used to retrieve list of existing licence files based
	 * on existence (or not) of particular field.
	 *
	 * @param property field which will be checked in the licence file
	 * @param exists whether return files that DO contain or DO NOT contain mentioned property
	 * @return {@code List} of {@code Licence} files
	 * @throws NoSuchAlgorithmException
	 * @throws InvalidKeySpecException
	 */
	private List<Licence> getLicenceFiles(String property, boolean exists)
			throws NoSuchAlgorithmException, InvalidKeySpecException {
		List<Licence> result = new ArrayList<>();
		final Path etcDir = Paths.get("./etc/");
		if (!etcDir.toFile().exists()) {
			return null;
		}
		try (final DirectoryStream<Path> licences = Files.newDirectoryStream(etcDir, "*.licence")) {
			LicenceLoader loader = new LicenceLoaderImpl();
			for (Path lic : licences) {
				final Licence licence = loader.loadLicence(lic.toFile());
				final String propertyAsString = licence.getPropertyAsString(property);
				if (exists) {
					if (propertyAsString != null) {
						result.add(licence);
					}
				} else {
					if (propertyAsString == null) {
						result.add(licence);
					}
				}
			}
		} catch (IOException e) {
			log.log(Level.WARNING, "There was a problem reading current licence files", e);
		}
		return result;
	}

	protected Element loadData(final String url, final List<JID> managedVHosts, List<String> oldLicencesHashes)
			throws TooManyRequestsHTTPException, GeneralSecurityException, IOException {
		String encodedData = prepareRequest(managedVHosts, oldLicencesHashes);
		String u = (url == null ? DEFAULT_INSTALLATION_ID_SERVER_URL : url);
		return dl.loadData(u, encodedData);
	}

	private String prepareRequest(final List<JID> managedVHosts, final List<String> oldLicencesHashes) {
		StringBuffer sb = new StringBuffer();
		sb.append("<data><fields>");
		DataLoader.addRequestItems(sb, "vhosts", managedVHosts);
		DataLoader.addRequestItems(sb, "legacy-hash", oldLicencesHashes);

		TOTP t = new TOTP();
		sb.append("<item><var>totp</var><value>").append(t.generateTOTP()).append("</value></item>");

		sb.append("</fields></data>");

		return sb.toString();
	}

	private String retrieveInstallationId() throws TooManyRequestsHTTPException, IOException {
		String instId = null;
		// retrieve installation-id from database
		try {

			String resource = System.getProperty(RepositoryFactory.GEN_USER_DB_URI_PROP_KEY);
			UserRepository userRepository = RepositoryFactory.getUserRepository(null, resource, null);

			instId = userRepository.getData(licenceUserName, INSTALLATION_ID_KEY);
			if (log.isLoggable(Level.FINEST)) {
				log.log(Level.FINEST, "Retrieved installationID from database: {0}", new Object[]{instId});
			}
			if (instId != null) {
				if (log.isLoggable(Level.WARNING)) {
					log.log(Level.WARNING, "Retrieved installationID from database: {0}", new Object[]{instId});
				}
				return instId;
			}
		} catch (Exception e) {
			log.log(Level.WARNING, "There was a problem accessing repository to retrieve ID",
			        e.getMessage());
			if (log.isLoggable(Level.FINEST)) {
				log.log(Level.FINEST, "There was a problem accessing repository to retrieve ID", e);
			}

			return null;
		}

		// ok, so installation-id is not yet in database

		if (INSTALLATION_ID_FILE.toFile().exists() && INSTALLATION_ID_FILE.toFile().length() != 0) {
			// however we already have installation-id file, so this installation
			// already retrieved installation-id! Let's inform user about it but
			// not assume it's correct - it may have been a test, better instruct
			// user what to do.

			Scanner sc = new Scanner(INSTALLATION_ID_FILE);
			StringBuilder sb = new StringBuilder();
			while (sc.hasNextLine() ) {
				sb.append(sc.nextLine());
			}

			log.log(Level.WARNING,
					"\n\n\n  =====================================\n"
					+ "  You already obtained installation-id: " + sb.toString() + "\n"
					+ "  However it's not stored in the database which may result in cluster\n"
					+ "  not obtaining correct licences!\n"
					+ "  \n"
					+ "  Please make sure that database and cluster is working correctly!\n"
					+ "  =====================================\n\n");

			instId = sb.toString();
		}


		String installationIdFromInitProperties = System.getProperty(INSTALLATION_ID_KEY);

//		if (instId == null || !instId.equals(installationIdFromFile)) {
		// retrieve installation-id from file, overwrite in database with the one from file if different

		if (instId == null && installationIdFromInitProperties != null) {
			if (log.isLoggable(Level.FINEST)) {
				log.log(Level.FINEST, "Retrieved installationID from file: {0}", new Object[]{installationIdFromInitProperties});
			}

			instId = installationIdFromInitProperties;
		}

		if (instId == null) {
			try {
				// ok, so we don't have installation id neither in file nor in DB - let's grab one from server
				final String idFromServer;
				idFromServer = retrieveInstallationIdFromServer();
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST, "Retrieved installationID from server: {0}", new Object[]{idFromServer});
				}

				if (idFromServer != null && !idFromServer.isEmpty()) {
					instId = idFromServer;
				}
			} catch (GeneralSecurityException e) {
				log.log(Level.WARNING, "There was a problem retrieving installation-id from the server", e);
			}
		}

		if (instId == null) {
			// if we couldn't retrieve installation-id with previous methods
			// and if there are existing licences and all have same installation-id
			boolean valid = false;
			String id = null;

			try {
				final List<Licence> licenceFiles = getLicenceFiles("installation-id", true);
				if (licenceFiles != null && !licenceFiles.isEmpty()) {
					for (Licence licence : licenceFiles) {

						if (licence != null) {
							final String idFromProperty = licence.getPropertyAsString("installation-id");
							if (idFromProperty != null) {
								if (id == null || id.equals(idFromProperty)) {
									id = idFromProperty;
									valid = true;
								} else if (!id.equals(idFromProperty)) {
									valid = false;
								}
							}
						}
					}
					if (valid && id != null) {
						instId = id;
					}
				}
			} catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
				log.log(Level.WARNING, "There was a problem retrieving installation-id from existing files", e);
			}
		}

		if (instId != null) {
			// store installationId to database
			try {
				String resource = System.getProperty(RepositoryFactory.GEN_USER_DB_URI_PROP_KEY);
				UserRepository userRepository = RepositoryFactory.getUserRepository(null, resource, null);
				userRepository.setData(licenceUserName, INSTALLATION_ID_KEY, instId);
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST, "Stored installationID in database: {0}", resource);
				}

			} catch (Exception e) {
				log.log(Level.WARNING, "There was a problem accessing repository to store ID", e.getMessage());
			}
			// store to file

			try {
				if (!INSTALLATION_ID_FILE.getParent().toFile().exists()) {
					INSTALLATION_ID_FILE.getParent().toFile().mkdirs();
				}

				Files.write(INSTALLATION_ID_FILE, instId.getBytes());
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST, "Stored installationID in file: {0}",
					        new Object[]{INSTALLATION_ID_FILE.toAbsolutePath()});
				}

			} catch (IOException ex) {
				log.log(Level.WARNING, "There was a problem writing ID to a file", ex);
			}

			log.log(Level.WARNING,
			        "Your installation-id is: {0} and it has been written to database and to the file: {1}",
			        new Object[]{instId, INSTALLATION_ID_FILE.toAbsolutePath()});
		} else {
			log.log(Level.WARNING,
			        "There was a problem obtaining installation-id! Please read 'Licensing' chapter from Tigase Administration Guide for more information how to obtain installation-id");
		}
		return instId;
	}

	private String retrieveInstallationIdFromServer()
			throws TooManyRequestsHTTPException, GeneralSecurityException, IOException {
		List<String> digests = new ArrayList<>();
		List<Licence> licences = getLicenceFiles("licence-nr", false);
		if (licences != null && !licences.isEmpty()) {
			licences.forEach((licence) -> digests.add(licence.getLicenceDigest()));
		}

		if (!digests.isEmpty()) {
			log.log(Level.WARNING,
			        "Legacy licences present, if you want to obtain installation-id manually please include following digest codes in the form: {0})", digests);
		}

		final Element element = loadData(null, VHostsRetriever.getManagedVHosts(), digests);
		if (element != null) {
			final String installationID = element.getCDataStaticStr(new String[]{"command", "fields", "item", "value"});
			log.log(Level.FINE, "Retrieved installationId: {0} from request: {1}",
			        new Object[]{installationID, element});
			return installationID;
		} else {
			return null;
		}
	}

}
