/*
 * Tigase Jabber/XMPP Server
 * Copyright (C) 2004-2013 "Tigase, Inc." <office@tigase.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. Look for COPYING file in the top folder.
 * If not, see http://www.gnu.org/licenses/.
 *
 */
package tigase.licence;

//~--- non-JDK imports --------------------------------------------------------
import tigase.licence.LicenceValidator.ValidationResult.*;

import tigase.stats.collector.*;
import tigase.stats.collector.provider.StatisticsUploader;

import tigase.xml.Element;

//~--- JDK imports ------------------------------------------------------------
import java.io.File;
import java.io.IOException;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.TimerTask;
import java.util.logging.FileHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.crypto.spec.SecretKeySpec;

/**
 * Class used to perform periodic licence check and/or upload of statistics
 * data.
 * <p>
 * @author Wojciech Kapcia <wojciech.kapcia@tigase.org>
 */
public class LicenceChecker {

	/** Number of failed attempts to send the statistics */
	protected static int failCount = 0;
	/** Limit number of all possible attempts to send statistics */
	protected final static int failCountLimit = 10;
	/** Logger object */
	private static final Logger log = Logger.getLogger( "tigase.stats" );
	/** Holds secret used to to sign uploaded data */
	private static final String SECRET = "0000";
	/** Holds key based on secret used to sign uploaded data */
	private static final SecretKeySpec key = new SecretKeySpec( SECRET.getBytes(), "HmacSHA1" );
	/** DefaultElementSigner used to sing uploaded data */
	protected static final ElementSigner signer = new DefaultElementSigner( key );
	/** Uploader object using created signer to upload data */
	protected static StatisticsUploader uploader = new StatisticsUploader( signer );
	/** Callback object to handle responses from server */
	private static StatisticsUploader.ResultCallback resCall;
	/** Executor service handling timed execution of licence checking */
	protected static ScheduledExecutorService timer = Executors.newSingleThreadScheduledExecutor();
	/** Timer task responsible for licence checking */
	protected static TimerTask licencePeriodicCheck;
	/** Stores component name for which licence check is performed */
	private final String componentName;
	/** Holds licence object */
	private Licence lic;
	/** Holds licence file */
	private final File LICENCE_FILE;
	/** Callback used to update statistics data in component */
	private final LicenceCheckerUpdateCallback updateCall;
	/** Holds date until which licence is valid */
	private Date validUntil;
//	/** performs daily checks */
//	private boolean doDaily = true;
	/** indicates whether it's the first run of the checker */
	private boolean firstCheck = true;
	/**
	 * Factory methods for {@link tigase.licence.LicenceChecker}.
	 */
	protected static final Map<String, LicenceChecker> licenceCheckers = new HashMap<String, LicenceChecker>( 5 );

	/**
	 * Creates a {@link tigase.licence.LicenceChecker} with default update
	 * callback {@link tigase.licence.LicenceCheckerUpdateCallbackImpl} for the
	 * component which name was provided as parameter. Component name will be used
	 * to find licence file with the pattern of
	 * <code>etc/${cmpName}.licence</code>
	 * <p>
	 * @param cmpName name of the component for which
	 *                {@link tigase.licence.LicenceChecker} will be instantiated.
	 * @return {@link tigase.licence.LicenceChecker} for the given cmpName
	 */
	public static LicenceChecker getLicenceChecker( String cmpName ) {
		return getLicenceChecker( cmpName, null );
	}

	/**
	 * Creates a {@link tigase.licence.LicenceChecker} with custom implementation
	 * of {@link tigase.licence.LicenceCheckerUpdateCallback} interface for the
	 * component which name was provided as parameter. Component name will be used
	 * to find licence file with the pattern of
	 * <code>etc/${cmpName}.licence</code>
	 * <p>
	 * @param cmpName name of the component for which
	 *                {@link tigase.licence.LicenceChecker} will be instantiated.
	 * @param call    custom implementation of
	 *                {@link tigase.licence.LicenceCheckerUpdateCallback}
	 * @return {@link tigase.licence.LicenceChecker} for the given cmpName
	 */
	public static LicenceChecker getLicenceChecker( String cmpName,
																									LicenceCheckerUpdateCallback call ) {

		if ( call == null ){
			call = new LicenceCheckerUpdateCallbackImpl( cmpName );
		}

		LicenceChecker licenceChecker = licenceCheckers.get( cmpName );

		if ( licenceChecker == null ){
			licenceChecker = new LicenceChecker( cmpName, call );
			licenceCheckers.put( cmpName, licenceChecker );
		}

		return licenceChecker;

	}
	//~--- constructors ---------------------------------------------------------

	/**
	 * Constructs new LicenceChecker object which entails creation of new licence
	 * file representation. If there wasn't one previously, a new timer task,
	 * which performs periodic licence check is created and first check is
	 * scheduled 5 minutes after startup.
	 * <p>
	 *
	 * @param cmpName name of the component for which licence checker was created
	 * @param call    callback object used to update statistics data in component
	 */
	private LicenceChecker( String cmpName, final LicenceCheckerUpdateCallback call ) {
		this.updateCall = call;
		this.componentName = cmpName;
		setLogger();

		// create only one daily licence checker task and schedule first check
		// 5 minutes after startup.
		if ( licencePeriodicCheck == null ){
			licencePeriodicCheck = new LicencePeriodicCheckTask();
			timer.schedule( licencePeriodicCheck, 5, TimeUnit.MINUTES );
		}

		log.log( Level.CONFIG, "Created licence checker for {0}", cmpName );
		LICENCE_FILE = new File( "etc/" + componentName + ".licence" );
	}

	//~--- methods --------------------------------------------------------------
	/**
	 * Performs actual verification of licence validity
	 * <p>
	 * @return <strong>true</strong> for valid licence, <strong>false</strong>
	 * otherwise
	 */
	private boolean validateLicence() {
		try {
			final LicenceLoader loader = LicenceLoaderFactory.create();

			if ( !LICENCE_FILE.exists() ){
				log.log( Level.WARNING, "Missing licence file ({0})!", LICENCE_FILE );

				return false;
			}
			lic = loader.loadLicence( LICENCE_FILE );
			validUntil = lic.getPropertyAsDate( Licence.VALID_UNTIL_KEY );
			try {
				switch ( lic.check() ) {
					case invalidDates:
						log.warning( "Licence is expired." );
						return false;

					case invalidSignature:
						log.warning( "Invalid or modified licence file." );
						return false;

					case valid:
						log.warning( "Licence OK" );
						break;
				}
			} catch ( Exception e ) {
				log.log( Level.WARNING, "Licence invalid", e );
				return false;
			}

			String appId = lic.getPropertyAsString( "app-id" );
			if ( ( appId == null ) || !appId.equals( componentName ) ){
				log.log( Level.WARNING, "This is not licence for {0} Component!", componentName );
				return false;
			}
		} catch ( Exception e ) {
			log.log( Level.WARNING, "Can't load licence file. Error: {0}", e.getMessage() );
			return false;
		}
		return true;
	}

	//~--- get methods ----------------------------------------------------------
	/**
	 * Returns date until which licence is valid.
	 *
	 * @return a <doce>Date</code> representation of licence expiration date.
	 */
	public Date getValidUntil() {
		return validUntil;
	}

	/**
	 * Return callback used to update statistics data for particular
	 * licenceChecker
	 *
	 * @return callback used to update statistics data for particular
	 *         licenceChecker
	 */
	public LicenceCheckerUpdateCallback getUpdateCall() {
		return updateCall;
	}

	//~--- set methods ----------------------------------------------------------
	/**
	 * Performs logger setup - all statistics related logging is also written to
	 * separate log file.
	 * <p>
	 * @throws SecurityException
	 */
	private void setLogger() throws SecurityException {
		Logger lic_log = Logger.getLogger( "tigase.licence" );
		log.setLevel( Level.ALL );
		lic_log.setLevel( Level.ALL );
		try {
			Handler filehandler = new FileHandler( "logs/statistics.log" );

			filehandler.setLevel( Level.ALL );
			lic_log.addHandler( filehandler );
			log.addHandler( filehandler );
		} catch ( IOException ex ) {
			log.log( Level.CONFIG, componentName );
		}

//  Handler handler = new ConsoleHandler();
//  handler.setLevel( Level.ALL );
//  log.addHandler( handler );
	}

	//~--- inner classes --------------------------------------------------------
	/**
	 * Timer task responsible for license checking.
	 * <p>
	 * Following scheme is used:
	 * <ul>
	 * <li>in case of valid license next check is schedule after validity
	 * expired</li>
	 * <li>in case of invalid or missing license StatisticsData is collected and
	 * an attempt to upload it to Tigase REST server is made; following additional
	 * conditions apply:
	 * <ul>
	 * <li>if this is a cold start of the server and thus a first attempt at
	 * statistics generation and upload a complete fail to upload it resulting in
	 * StatisticsUploader executing {@code .onFailure()} callback of
	 * {@link tigase.stats.collector.provider.StatisticsUploader.ResultCallback}
	 * object (i.e. after 30 attempt, each scheduled twice longer than the
	 * previous, amounting to around 15 hours) then an immediate system shutdown
	 * is performed.</li>
	 * <li>if there were an successful submission of statistics then this is only
	 * a temporary issue with uploading statistics then a subsequent check is
	 * schedule with 1 day delay and we allow 10 days of lax treatment; in case of
	 * 10 failed attempts to upload data (in the above step) a system shutdown is
	 * performed</li>
	 * </ul>
	 * </ul>
	 */
	private class LicencePeriodicCheckTask extends TimerTask {

		@Override
		public void run() {
			boolean invalidLicencePresent = false;
			long validUntilDiff = Long.MAX_VALUE;
			long now = new Date().getTime();

			log.log( Level.WARNING, "\n\nLicense verification runtime.\n" );

			if ( failCount > failCountLimit ){
				log.log( Level.SEVERE,
								 "Could not upload statistics within required period of time, shutting down the system" );
				System.exit( 402 );
			}

			Element additionalData = new Element( "unlicencedComponenents" );

			// Check all licence checkers.
			for ( String key : licenceCheckers.keySet() ) {
				LicenceChecker licChecker = licenceCheckers.get( key );
				if ( licChecker.validateLicence() ){
					log.log( Level.WARNING, "Licence valid untill: {0}", licChecker.getValidUntil() );

					// calculate earliest time next check should be performed
					long nextCheck = licChecker.validUntil.getTime() - now;
					validUntilDiff = nextCheck < validUntilDiff ? nextCheck : validUntilDiff;

				} else {
					// licence is not valid, grab additional information from component
					// (if provieded)
					log.warning( licChecker.getUpdateCall().getMissingLicenseWarning() );
					Element cmpAdditionalData = licChecker.getUpdateCall().getComponentAdditionalData();
					additionalData.addChild( cmpAdditionalData );

					invalidLicencePresent |= true;
				}
			}

			// if at least one licene is invalid then upload data and schedule
			// next check in one 1 day
			if ( invalidLicencePresent ){
				StatisticsData data;
				data = LicenceCheckerUpdater.updateData();
				// concat additionalData
				data.setAdditionalData( data.getAdditionalData() + additionalData.toString() );

				log.warning( "Uploading data: " + data.toElement() );
				resCall = new ResCall();
				uploader.upload( data, resCall );

				timer.schedule( licencePeriodicCheck, 1, TimeUnit.DAYS );
			} else {
				// all licences were valid, we schedule next check at the ealiest time
				// any of the licences would be invalid
				timer.schedule( licencePeriodicCheck, validUntilDiff, TimeUnit.MILLISECONDS );
			}
			log.log( Level.WARNING, "\n\n" );
		}
	}

	/**
	 * Implementation for handling results of uploading statistics. On failed
	 * attempt either the system is shut down (if it was a cold run, i.e. the
	 * first one) or the counter of fails is increased and on success attempt the
	 * counter is reset and the system allows for less strict enforcement of
	 * failed statistics upload.
	 */
	public class ResCall
			implements StatisticsUploader.ResultCallback {

		/**
		 * <code>onFailure</code> callback. If it's called on the cold run (i.e.
		 * first execution) then immediately shut down the system; otherwise if
		 * there were previously successful attempts to submit data we can assume
		 * that this is not intentional thus allow less strict enforcement and run
		 * the check for the consecutive 10 days each time resulting in increase of
		 * failed attempts counter.
		 * <p>
		 *
		 * @param ex is a <code>Exception</code>
		 */
		@Override
		public void onFailure( Exception ex ) {
			if ( firstCheck ){
				log.log( Level.SEVERE,
								 "Could not upload statistics within required period of time, shutting down the system" );
				System.exit( 402 );
			} else {
				failCount++;
				log.log( Level.WARNING, "Could not upload statistics; "
																+ ( failCountLimit - failCount ) + " attempts before system shutdown", ex );
			}
		}

		/**
		 * <code>onSuccess</code> callback resulting in reset of failed attempts
		 * counter as well as switching to less strict licence checking which allows
		 * running component for 10 days without successful execution.
		 * <p>
		 *
		 * @param result is a <code>Element</code>
		 */
		@Override
		public void onSuccess( Element result ) {
			log.log( Level.WARNING, "statistics uploaded = {0}; resetting number of attempts",
							 ( ( result != null )
								 ? result.toString()
								 : null ) );
			// reset number of attempts
			failCount = 0;
			// run checking in regular mode (i.e. allow 10 days of failed submission
			// before shutting down the system) 
			firstCheck = false;
		}
	}
}


//~ Formatted in Tigase Code Convention on 13/10/17
