package tigase.licence;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Serializable;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SignatureException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import tigase.licence.LicenceValidator.ValidationResult;

final class ImmutableLicence implements Licence, Serializable {

	private static final class Entry implements Serializable {

		private static final long serialVersionUID = 8338663780387437875L;

		private boolean equalSign;

		private String key;

		private String value;

		public byte[] getBytes() {
			String out = key + "=" + (value == null ? "" : value) + new String(new byte[] { 0x0a });
			return out.getBytes();
		}

		public boolean isEqualSign() {
			return equalSign;
		}

		public void setEqualSign(boolean equalSign) {
			this.equalSign = equalSign;
		}

		public void setKey(String key) {
			this.key = key;
		}

		public void setValue(String value) {
			this.value = value;
		}
	}

	private static final String DATE_FORMAT = "yyyy-MM-dd";

	private static final long serialVersionUID = 1L;

	private List<Entry> entries = new ArrayList<Entry>();

	private Map<String, Entry> entriesMap = new HashMap<String, Entry>();

	private LicenceValidatorImpl validator;

	ImmutableLicence() {
	}

	@Override
	public ValidationResult check() throws InvalidKeyException, NoSuchAlgorithmException, SignatureException, ParseException {
		return this.validator.check(this);
	}

	@Override
	public synchronized byte[] getBytes() {
		StringBuilder sb = new StringBuilder();
		for (Entry entry : this.entries) {
			if (entry.key == null || entry.key.startsWith("#") || !entry.isEqualSign()) {
				continue;
			}
			if ("signature".equals(entry.key)) {
				continue;
			}
			sb.append(new String(entry.getBytes()));
		}
		return sb.toString().getBytes();
	}

	private String getProperty(String key) {
		Entry entry = this.entriesMap.get(key);
		if (entry == null) {
			return null;
		} else {
			return entry.value;
		}
	}

	@Override
	public Boolean getPropertyAsBoolean(String key) {
		String value = getProperty(key);
		return value == null ? null : value.trim().equalsIgnoreCase("yes");
	}

	@Override
	public Calendar getPropertyAsCalendar(String key) throws ParseException {

		String value = getProperty(key);
		Calendar cal = Calendar.getInstance();
		final DateFormat DATE_FORMATTER = new SimpleDateFormat(DATE_FORMAT);

		Date dt = value == null ? null : DATE_FORMATTER.parse(value);
		if (dt != null) {
			cal.setTime(dt);
		} else
			cal = null;
		return cal;
	}

	@Override
	public Date getPropertyAsDate(String key) throws ParseException {
		String value = getProperty(key);
		final DateFormat dateFormatter = new SimpleDateFormat(DATE_FORMAT);

		return value == null ? null : dateFormatter.parse(value);
	}

	@Override
	public Double getPropertyAsDouble(String key) {
		String value = getProperty(key);
		return value == null ? null : Double.valueOf(value);
	}

	@Override
	public Integer getPropertyAsInteger(String key) {
		String value = getProperty(key);
		return value == null ? null : Integer.valueOf(value);
	}

	@Override
	public String getPropertyAsString(String key) {
		return getProperty(key);
	}

	void load(InputStream in) throws IOException {
		BufferedReader reader = new BufferedReader(new InputStreamReader(in));

		entries.clear();
		entriesMap.clear();

		String line;
		while ((line = reader.readLine()) != null) {
			put(line);
		}

		reader.close();
	}

	private void put(String line) {
		if (line == null)
			return;
		int p = line.indexOf("=");

		String key;
		String value;
		if (p > 0) {
			key = line.substring(0, p).trim();
			value = line.substring(p + 1).trim();
		} else {
			key = line;
			value = null;
		}

		Entry entry = new Entry();
		entry.setKey(key);
		entry.setValue(value);
		entry.setEqualSign(p != -1);
		entries.add(entry);
		entriesMap.put(key, entry);
	}

	void setValidator(LicenceValidatorImpl licenceValidatorImpl) {
		this.validator = licenceValidatorImpl;
	}

}
