/*
 * SocketConnector.java
 *
 * Tigase XMPP Client Library
 * Copyright (C) 2004-2018 "Tigase, Inc." <office@tigase.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. Look for COPYING file in the top folder.
 * If not, see http://www.gnu.org/licenses/.
 */
package tigase.jaxmpp.j2se.connectors.socket;

import org.bouncycastle.tls.*;
import org.bouncycastle.tls.crypto.TlsCertificate;
import org.bouncycastle.tls.crypto.impl.bc.BcTlsCrypto;
import tigase.jaxmpp.core.client.*;
import tigase.jaxmpp.core.client.Connector.ConnectedHandler.ConnectedEvent;
import tigase.jaxmpp.core.client.Connector.EncryptionEstablishedHandler.EncryptionEstablishedEvent;
import tigase.jaxmpp.core.client.Connector.ErrorHandler.ErrorEvent;
import tigase.jaxmpp.core.client.Connector.StanzaReceivedHandler.StanzaReceivedEvent;
import tigase.jaxmpp.core.client.Connector.StanzaSendingHandler.StanzaSendingEvent;
import tigase.jaxmpp.core.client.Connector.StateChangedHandler.StateChangedEvent;
import tigase.jaxmpp.core.client.SessionObject.Scope;
import tigase.jaxmpp.core.client.connector.StreamError;
import tigase.jaxmpp.core.client.eventbus.EventHandler;
import tigase.jaxmpp.core.client.eventbus.JaxmppEvent;
import tigase.jaxmpp.core.client.exceptions.JaxmppException;
import tigase.jaxmpp.core.client.factory.UniversalFactory;
import tigase.jaxmpp.core.client.xml.Element;
import tigase.jaxmpp.core.client.xml.ElementFactory;
import tigase.jaxmpp.core.client.xml.XMLException;
import tigase.jaxmpp.core.client.xmpp.modules.StreamFeaturesModule;
import tigase.jaxmpp.core.client.xmpp.modules.registration.InBandRegistrationModule;
import tigase.jaxmpp.core.client.xmpp.stanzas.Stanza;
import tigase.jaxmpp.core.client.xmpp.stanzas.StreamPacket;
import tigase.jaxmpp.j2se.DNSResolver;
import tigase.jaxmpp.j2se.connectors.socket.SocketConnector.HostChangedHandler.HostChangedEvent;

import javax.net.ssl.*;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.net.*;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.Deflater;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.Inflater;
import java.util.zip.InflaterInputStream;

/**
 *
 */
public class SocketConnector
		implements Connector {

	public final static String COMPRESSION_DISABLED_KEY = "COMPRESSION_DISABLED";
	public final static HostnameVerifier DEFAULT_HOSTNAME_VERIFIER = new DefaultHostnameVerifier();
	/**
	 * Default size of buffer used to decode data before parsing
	 */
	public final static int DEFAULT_SOCKET_BUFFER_SIZE = 2048;
	public static final String HOSTNAME_VERIFIER_DISABLED_KEY = "HOSTNAME_VERIFIER_DISABLED_KEY";
	public static final String HOSTNAME_VERIFIER_KEY = "HOSTNAME_VERIFIER_KEY";
	public static final String KEY_MANAGERS_KEY = "KEY_MANAGERS_KEY";
	public static final String SASL_EXTERNAL_ENABLED_KEY = "SASL_EXTERNAL_ENABLED_KEY";
	public static final String SERVER_HOST = "socket#ServerHost";
	public static final String SERVER_PORT = "socket#ServerPort";
	public static final String USE_PLAIN_SSL_KEY = "USE_PLAIN_SSL_KEY";
	/**
	 * Socket timeout.
	 */
	public static final int DEFAULT_SOCKET_TIMEOUT = 0; // 1000 * 60 * 3;
	public static final String SSL_SOCKET_FACTORY_KEY = "socket#SSLSocketFactory";
	public static final String TLS_DISABLED_KEY = "TLS_DISABLED";
	/**
	 * Property to specify custom {@link Socket#setSoTimeout(int) socket timeout} for SSL Socket. Default is
	 * <code>0</code>.
	 */
	public static final String SSL_SOCKET_TIMEOUT_KEY = "SSL_SOCKET_TIMEOUT_KEY";
	/**
	 * Property to specify custom {@linkplain Socket#setSoTimeout(int) socket timeout} for Plain Socket. Default is
	 * {@link SocketConnector#DEFAULT_SOCKET_TIMEOUT 180000 ms}.
	 */
	public static final String PLAIN_SOCKET_TIMEOUT_KEY = "PLAIN_SOCKET_TIMEOUT_KEY";
	public static final String KEEP_ALIVE_DELAY_KEY = "KEEP_ALIVE_DELAY_KEY";
	public static final String TLS_SESSION_ID_KEY = "TLS_SESSION_ID_KEY";
	public static final String TLS_PEER_CERTIFICATE_KEY = "TLS_PEER_CERTIFICATE_KEY";
	public static final String USE_BOUNCYCASTLE_KEY = "USE_BOUNCYCASTLE_KEY";
	private final static Charset UTF_CHARSET = StandardCharsets.UTF_8;
	/**
	 * Instance of empty byte array used to force flush of compressed stream
	 */
	private final static byte[] EMPTY_BYTEARRAY = new byte[0];
	private final static Object mutex = new Object();
	private static int counter = 0;
	private final Object ioMutex = new Object();
	private final Logger log;
	private final int oid;
	private Timer closeTimer;
	private Context context;
	private TimerTask pingTask;
	private volatile Reader reader;
	private Socket socket;
	private Timer timer;
	private Worker worker;
	private OutputStream writer;

	public static boolean isTLSAvailable(SessionObject sessionObject) throws XMLException {
		final Element sf = StreamFeaturesModule.getStreamFeatures(sessionObject);
		if (sf == null) {
			return false;
		}
		Element m = sf.getChildrenNS("starttls", "urn:ietf:params:xml:ns:xmpp-tls");
		return m != null;
	}

	/**
	 * Returns true if server send stream features in which it advertises support for stream compression using ZLIB
	 *
	 * @param sessionObject
	 *
	 * @return
	 *
	 * @throws XMLException
	 */
	public static boolean isZLibAvailable(SessionObject sessionObject) throws XMLException {
		final Element sf = StreamFeaturesModule.getStreamFeatures(sessionObject);
		if (sf == null) {
			return false;
		}
		Element m = sf.getChildrenNS("compression", "http://jabber.org/features/compress");
		if (m == null) {
			return false;
		}

		for (Element method : m.getChildren("method")) {
			if ("zlib".equals(method.getValue())) {
				return true;
			}
		}

		return false;
	}

	public SocketConnector(Context context) {
		synchronized (mutex) {
			oid = (++counter);
		}
		this.log = Logger.getLogger(this.getClass().getName());
		this.context = context;
	}

	@Override
	public XmppSessionLogic createSessionLogic(XmppModulesManager modulesManager, PacketWriter writer) {
		if (context.getSessionObject().getProperty(InBandRegistrationModule.IN_BAND_REGISTRATION_MODE_KEY) ==
				Boolean.TRUE) {
			printLog(Level.INFO, "Using XEP-0077 mode!!!!");
			return new SocketInBandRegistrationXmppSessionLogic(this, modulesManager, context);
		} else {
			return new SocketXmppSessionLogic(this, modulesManager, context);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public State getState() {
		if (this.context == null) {
			return State.disconnected;
		}
		State st = this.context.getSessionObject().getProperty(CONNECTOR_STAGE_KEY);
		return st == null ? State.disconnected : st;
	}

	/**
	 * Returns true when stream is compressed
	 *
	 * @return
	 */
	@Override
	public boolean isCompressed() {
		return context.getSessionObject().getProperty(COMPRESSED_KEY) == Boolean.TRUE;
	}

	@Override
	public boolean isSecure() {
		return context.getSessionObject().getProperty(ENCRYPTED_KEY) == Boolean.TRUE;
	}

	@Override
	public void keepalive() throws JaxmppException {
		if (context.getSessionObject().getProperty(DISABLE_KEEPALIVE_KEY) == Boolean.TRUE) {
			return;
		}
		if (getState() == State.connected) {
			send(new byte[]{32});
		}
	}

	public void onTLSStanza(Element elem) throws JaxmppException {
		if (elem.getName().equals("proceed")) {
			proceedTLS();
		} else if (elem.getName().equals("failure")) {
			printLog(Level.INFO, "TLS Failure");
		}
	}

	/**
	 * Handles result of requesting stream compression
	 *
	 * @param elem
	 *
	 * @throws JaxmppException
	 */
	public void onZLibStanza(Element elem) throws JaxmppException {
		if (elem.getName().equals("compressed") && "http://jabber.org/protocol/compress".equals(elem.getXMLNS())) {
			proceedZLib();
		} else if (elem.getName().equals("failure")) {
			printLog(Level.INFO, "ZLIB Failure");
		}
	}

	public void processElement(Element elem) throws JaxmppException {
		if (getState() == State.disconnected) {
			printLog(Level.FINEST, "Ignoring element: " + elem.getAsString());
			return;
		}
		if (log.isLoggable(Level.FINEST)) {
			printLog(Level.FINEST, "Recv : " + elem.getAsString());
		}
		if (elem != null && elem.getXMLNS() != null && elem.getXMLNS().equals("urn:ietf:params:xml:ns:xmpp-tls")) {
			onTLSStanza(elem);
		} else if (elem != null && elem.getXMLNS() != null &&
				"http://jabber.org/protocol/compress".equals(elem.getXMLNS())) {
			onZLibStanza(elem);
		} else {
			onResponse(elem);
		}
	}

	@Override
	public void restartStream() throws JaxmppException {
		StringBuilder sb = new StringBuilder();
		sb.append("<stream:stream ");

		final BareJID from = context.getSessionObject().getProperty(SessionObject.USER_BARE_JID);
		String to;
		Boolean seeOtherHost = context.getSessionObject().getProperty(SEE_OTHER_HOST_KEY);
		if (from != null && (seeOtherHost == null || seeOtherHost)) {
			to = from.getDomain();
			sb.append("from='").append(from.toString()).append("' ");
		} else {
			to = context.getSessionObject().getProperty(SessionObject.DOMAIN_NAME);
		}

		if (to != null) {
			sb.append("to='").append(to).append("' ");
		}

		sb.append("xmlns='jabber:client' ");
		sb.append("xmlns:stream='http://etherx.jabber.org/streams' ");
		sb.append("version='1.0'>");

		if (log.isLoggable(Level.FINEST)) {
			printLog(Level.FINEST, "Restarting XMPP Stream");
		}
		send(sb.toString().getBytes(UTF_CHARSET));
		context.getEventBus().fire(new StreamRestartedHandler.StreamRestaredEvent(context.getSessionObject()));
	}

	public void send(byte[] buffer) throws JaxmppException {
		synchronized (ioMutex) {
			if (writer != null) {
				try {
					if (log.isLoggable(Level.FINEST)) {
						printLog(Level.FINEST, "Send : " + new String(buffer));
					}
					writer.write(buffer);
					writer.flush();
				} catch (IOException e) {
					throw new JaxmppException(e);

				}
			}
		}
	}

	@Override
	public void send(Element stanza) throws JaxmppException {
		synchronized (ioMutex) {
			if (writer != null) {
				try {
					String t = stanza.getAsString();
					if (log.isLoggable(Level.FINEST)) {
						printLog(Level.FINEST, "Send : " + t);
					}

					try {
						context.getEventBus().fire(new StanzaSendingEvent(context.getSessionObject(), stanza));
					} catch (Exception e) {
					}
					writer.write(t.getBytes(UTF_CHARSET));
				} catch (IOException e) {
					terminateAllWorkers();
					throw new JaxmppException(e);
				}
			}
		}
		try {
			Thread.sleep(2);
		} catch (InterruptedException e) {
			printLog(Level.WARNING, "Thread can't sleep. Insomnia?");
		}
	}

	@Override
	public void start() throws JaxmppException {
		setStage(State.connecting);
		if (timer != null) {
			try {
				timer.cancel();
			} catch (Exception e) {
			}
		}
		timer = new Timer("SocketConnectorTimer", true);

		if (context.getSessionObject().getProperty(HOSTNAME_VERIFIER_DISABLED_KEY) == Boolean.TRUE) {
			context.getSessionObject().setProperty(HOSTNAME_VERIFIER_KEY, null);
		} else if (context.getSessionObject().getProperty(HOSTNAME_VERIFIER_KEY) == null) {
			context.getSessionObject().setProperty(HOSTNAME_VERIFIER_KEY, DEFAULT_HOSTNAME_VERIFIER);
		}

		try {
			ArrayList<Entry> hosts = new ArrayList<>();
			Entry serverHost = getHostFromSessionObject();
			if (serverHost != null) {
				printLog(Level.INFO, "DNS entry stored in session object: " + serverHost);
				hosts.add(serverHost);
			}
			if (hosts.isEmpty()) {
				String x = context.getSessionObject().getProperty(SessionObject.DOMAIN_NAME);
				printLog(Level.INFO, "Resolving SRV record of domain '" + x + "'");
				DnsResolver dnsResolver = UniversalFactory.createInstance(DnsResolver.class.getName());
				if (dnsResolver != null) {
					if (log.isLoggable(Level.FINE)) {
						printLog(Level.FINE, "Using resolver provided by user: " + dnsResolver);
					}
					hosts.addAll(dnsResolver.resolve(x));
				} else {
					if (log.isLoggable(Level.FINE)) {
						printLog(Level.FINE, "Using built-in resolver");
					}
					hosts.addAll(DNSResolver.resolve(x));
				}

			}

			context.getSessionObject().setProperty(Scope.stream, DISABLE_KEEPALIVE_KEY, Boolean.FALSE);

			if (log.isLoggable(Level.FINER)) {
				printLog(Level.FINER, "Preparing connection to " + hosts);
			}

			for (Entry host : hosts) {
				try {
					socket = createSocket(host);
					break;
				} catch (java.net.NoRouteToHostException | UnknownHostException e) {
					printLog(Level.FINE, e.getMessage() + ". Trying next.");
				}
			}

			if (socket == null) {
				throw new JaxmppException("Cannot create socket.");
			}

			writer = socket.getOutputStream();
			reader = new TextStreamReader(socket.getInputStream());
			worker = new Worker(this) {

				@Override
				protected Reader getReader() {
					return SocketConnector.this.reader;
				}

				@Override
				protected void onErrorInThread(Exception e) throws JaxmppException {
					SocketConnector.this.onErrorInThread(e);
				}

				@Override
				protected void onStreamStart(Map<String, String> attribs) {
					SocketConnector.this.onStreamStart(attribs);
				}

				@Override
				protected void onStreamTerminate() throws JaxmppException {
					SocketConnector.this.onStreamTerminate();
				}

				@Override
				protected void processElement(Element elem) throws JaxmppException {
					SocketConnector.this.processElement(elem);
				}

				@Override
				protected void workerTerminated() {
					SocketConnector.this.workerTerminated(this);
				}

			};
			printLog(Level.FINEST, "Starting worker...");

			Boolean plainSSL = context.getSessionObject().getProperty(USE_PLAIN_SSL_KEY);
			if (plainSSL != null && plainSSL) {
				proceedTLS();
				worker.start();
			} else {
				worker.start();
				restartStream();
			}

			setStage(State.connected);

			this.pingTask = new TimerTask() {

				@Override
				public void run() {
					Thread t = new Thread() {
						@Override
						public void run() {
							try {
								keepalive();
							} catch (JaxmppException e) {
								printLog(Level.SEVERE, "Can't ping!", e);
							}
						}
					};
					t.setDaemon(true);
					t.start();
				}
			};

			if (context.getSessionObject().getProperty(EXTERNAL_KEEPALIVE_KEY) == null ||
					((Boolean) context.getSessionObject().getProperty(EXTERNAL_KEEPALIVE_KEY) == false)) {
				Integer defaultDelay = getTimeout(PLAIN_SOCKET_TIMEOUT_KEY, DEFAULT_SOCKET_TIMEOUT);
				defaultDelay = defaultDelay == null ? -1 : defaultDelay - 1000 * 5;

				Integer delay = getTimeout(KEEP_ALIVE_DELAY_KEY, defaultDelay);

				if (log.isLoggable(Level.CONFIG)) {
					printLog(Level.CONFIG, "Whitespace ping period is setted to " + delay + "ms");
				}

				if (delay != null) {
					timer.schedule(pingTask, delay, delay);
				}
			}

			fireOnConnected(context.getSessionObject());
		} catch (Exception e) {
			terminateAllWorkers();
			//onError(null, e);
			throw new JaxmppException(e);
		}
	}

	public void startTLS() throws JaxmppException {
		if (writer != null) {
			try {
				printLog(Level.FINE, "Start TLS ");
				Element e = ElementFactory.create("starttls", null, "urn:ietf:params:xml:ns:xmpp-tls");
				send(e);
			} catch (Exception e) {
				throw new JaxmppException(e);
			}
		}
	}

	/**
	 * Sends <compress/> stanza to start stream compression using ZLIB
	 *
	 * @throws JaxmppException
	 */
	public void startZLib() throws JaxmppException {
		if (writer != null) {
			try {
				printLog(Level.FINE, "Start ZLIB ");
				Element e = ElementFactory.create("compress", null, "http://jabber.org/protocol/compress");
				e.addChild(ElementFactory.create("method", "zlib", null));
				send(e);
			} catch (Exception e) {
				throw new JaxmppException(e);
			}
		}
	}

	@Override
	public void stop() throws JaxmppException {
		if (getState() == State.disconnected) {
			printLog(Level.FINE, "Ignoring stop connector.");
			return;
		}
		setStage(State.disconnecting);
		try {
			terminateStream();
		} catch (Exception e) {
			printLog(Level.WARNING, "Problem on terminating stream", e);
			setStage(State.disconnected);
		} finally {
			terminateAllWorkers();
		}
	}

	@Override
	@Deprecated
	public void stop(boolean terminate) throws JaxmppException {
		if (terminate) {
			printLog(Level.FINEST, "Terminating all workers immediately ");
			try {
				if (this.pingTask != null) {
					this.pingTask.cancel();
					this.pingTask = null;
				}
				closeSocket();
			} finally {
				setStage(State.disconnected);
				context = null;
			}
		} else {
			stop();
		}
	}

	protected void fireOnConnected(SessionObject sessionObject) throws JaxmppException {
		if (getState() == State.disconnected) {
			printLog(Level.WARNING, "Ignoring ConnectedEvent (WTF?)");
			return;
		}

		context.getEventBus().fire(new ConnectedEvent(sessionObject));
	}

	protected void fireOnError(Element response, Throwable caught, SessionObject sessionObject) throws JaxmppException {
		if (getState() == State.disconnected) {
			printLog(Level.FINEST, "Ignoring error: " + (response != null ? response.getAsString() : ""), caught);
			return;
		}
		StreamError streamError = null;
		if (response != null) {
			List<Element> es = response.getChildrenNS("urn:ietf:params:xml:ns:xmpp-streams");
			if (es != null) {
				for (Element element : es) {
					String n = element.getName();
					streamError = StreamError.getByElementName(n);
					break;
				}
			}
		}

		context.getEventBus().fire(new ErrorEvent(sessionObject, streamError, caught));
	}

	protected void fireOnStanzaReceived(StreamPacket response, SessionObject sessionObject) throws JaxmppException {
		context.getEventBus().fire(new StanzaReceivedEvent(sessionObject, response));
	}

	protected void fireOnTerminate(SessionObject sessionObject) throws JaxmppException {
		context.getEventBus().fire(new StreamTerminatedHandler.StreamTerminatedEvent(sessionObject));
	}

	protected String getAuthType(TlsKeyExchange tlsKeyExchange) {
		try {
			Field keyExchangeField = AbstractTlsKeyExchange.class.getDeclaredField("keyExchange");
			keyExchangeField.setAccessible(true);
			Object v = keyExchangeField.get(tlsKeyExchange);
			final int i = Integer.valueOf(v.toString()).intValue();
			switch (i) {
				case 0:
					return "NULL";
				case 1:
					return "RSA";
				case 2:
					return "RSA_EXPORT";
				case 3:
					return "DHE_DSS";
				case 4:
					return "DHE_DSS_EXPORT";
				case 5:
					return "DHE_RSA";
				case 6:
					return "DHE_RSA_EXPORT";
				case 7:
					return "DH_DSS";
				case 8:
					return "DH_DSS_EXPORT";
				case 9:
					return "DH_RSA";
				case 10:
					return "DH_RSA_EXPORT";
				case 11:
					return "DH_anon";
				case 12:
					return "DH_anon_EXPORT";
				case 13:
					return "PSK";
				case 14:
					return "DHE_PSK";
				case 15:
					return "RSA_PSK";
				case 16:
					return "ECDH_ECDSA";
				case 17:
					return "ECDHE_ECDSA";
				case 18:
					return "ECDH_RSA";
				case 19:
					return "ECDHE_RSA";
				case 20:
					return "ECDH_anon";
				case 21:
					return "SRP";
				case 22:
					return "SRP_DSS";
				case 23:
					return "SRP_RSA";
				case 24:
					return "ECDHE_PSK";
				default:
					return "UNKNOWN " + i;
			}
		} catch (Throwable e) {
			e.printStackTrace();
			return null;
		}
	}

	protected String getHostname() {
		final String hostname;
		if (context.getSessionObject().getProperty(SessionObject.USER_BARE_JID) != null) {
			hostname = ((BareJID) context.getSessionObject().getProperty(SessionObject.USER_BARE_JID)).getDomain();
		} else if (context.getSessionObject().getProperty(SessionObject.DOMAIN_NAME) != null) {
			hostname = context.getSessionObject().getProperty(SessionObject.DOMAIN_NAME);
		} else {
			hostname = null;
		}
		return hostname;
	}

	protected KeyManager[] getKeyManagers() throws NoSuchAlgorithmException {
		KeyManager[] result = context.getSessionObject().getProperty(KEY_MANAGERS_KEY);
		return result == null ? new KeyManager[0] : result;
	}

	/**
	 * Returns timeout value.
	 *
	 * @param propertyName name of property
	 * @param defaultValue default value if property is <code>null</code>.
	 *
	 * @return timeout value or <code>null</code> if value is less than 0.
	 */
	protected Integer getTimeout(String propertyName, int defaultValue) {
		Integer v = context.getSessionObject().getProperty(propertyName);
		int result = v == null ? defaultValue : v.intValue();
		return result < 0 ? null : result;
	}

	protected void onError(Element response, Throwable caught) throws JaxmppException {
		if (response != null) {
			Element seeOtherHost = response.getChildrenNS("see-other-host", "urn:ietf:params:xml:ns:xmpp-streams");
			if (seeOtherHost != null) {
				if (log.isLoggable(Level.FINE)) {
					printLog(Level.FINE, "Received see-other-host=" + seeOtherHost.getValue());
				}
				reconnect(seeOtherHost.getValue());
				return;
			}
		}
		terminateAllWorkers();
		fireOnError(response, caught, context.getSessionObject());
	}

	protected void onErrorInThread(Exception e) throws JaxmppException {
		if (getState() == State.disconnected) {
			printLog(Level.FINE, "Ignoring error in thread", e);
			return;
		}
		terminateAllWorkers();
		fireOnError(null, e, context.getSessionObject());
	}

	protected void onResponse(final Element response) throws JaxmppException {
		synchronized (ioMutex) {
			if ("error".equals(response.getName()) && response.getXMLNS() != null &&
					response.getXMLNS().equals("http://etherx.jabber.org/streams")) {
				onError(response, null);
			} else {
				StreamPacket p;
				if (Stanza.canBeConverted(response)) {
					p = Stanza.create(response);
				} else {
					p = new StreamPacket(response) {
					};
				}
				p.setXmppStream(context.getStreamsManager().getDefaultStream());
				fireOnStanzaReceived(p, context.getSessionObject());
			}
		}
	}

	protected void onStreamStart(Map<String, String> attribs) {
		// TODO Auto-generated method stub
	}

	protected void onStreamTerminate() throws JaxmppException {
		if (getState() == State.disconnected) {
			printLog(Level.FINE, "Ignoring: Stream terminated");
			return;
		}
		setStage(State.disconnected);

		if (log.isLoggable(Level.FINE)) {
			printLog(Level.FINE, "Stream terminated");
		}

		terminateAllWorkers();
		fireOnTerminate(context.getSessionObject());

	}

	protected void proceedTLS() throws JaxmppException {
		if (context.getSessionObject().getProperty(USE_BOUNCYCASTLE_KEY) == Boolean.TRUE) {
			proceedBCTLS();
		} else {
			proceedJCETLS();
		}
	}

	/**
	 * Method activates stream compression by replacing reader and writer fields values and restarting XMPP stream
	 *
	 * @throws JaxmppException
	 */
	protected void proceedZLib() throws JaxmppException {
		printLog(Level.FINE, "Proceeding ZLIB");
		try {
			context.getSessionObject().setProperty(Scope.stream, DISABLE_KEEPALIVE_KEY, Boolean.TRUE);

			writer = null;
			reader = null;
			printLog(Level.FINE, "Start ZLIB compression");

			Deflater compressor = new Deflater(Deflater.BEST_COMPRESSION, false);
			try {
				// on Android platform Deflater has field named flushParm which
				// can force flushing data to socket for us
				Field f = compressor.getClass().getDeclaredField("flushParm");
				if (f != null) {
					f.setAccessible(true);
					f.setInt(compressor, 2); // Z_SYNC_FLUSH
					writer = new DeflaterOutputStream(socket.getOutputStream(), compressor);
				}
			} catch (NoSuchFieldException ex) {

				// if we do not have field we are on standard Java VM
				try {
					// try to create flushable DeflaterOutputStream but it
					// exists
					// only on Java 7 so we access it using reflection for
					// compatibility
					Constructor<DeflaterOutputStream> flushable = DeflaterOutputStream.class.getConstructor(
							OutputStream.class, Deflater.class, boolean.class);

					// we need wrap DeflaterOutputStream to flush it every time
					// we are
					// writing to it
					writer = new OutputStreamFlushWrap(
							flushable.newInstance(socket.getOutputStream(), compressor, true));

				} catch (NoSuchMethodException ex1) {
					// if we do not find constructor from Java 7 we use flushing
					// algorithm which was working fine on Java 6
					writer = new DeflaterOutputStream(socket.getOutputStream(), compressor) {
						@Override
						public void write(byte[] data) throws IOException {
							super.write(data);
							super.write(EMPTY_BYTEARRAY);
							super.def.setLevel(Deflater.NO_COMPRESSION);
							super.deflate();
							super.def.setLevel(Deflater.BEST_COMPRESSION);
							super.deflate();
						}
					};
				}
			}

			Inflater decompressor = new Inflater(false);
			final InflaterInputStream is = new InflaterInputStream(socket.getInputStream(), decompressor);
			reader = new TextStreamReader(is);

			context.getSessionObject().setProperty(Scope.stream, Connector.COMPRESSED_KEY, true);
			printLog(Level.INFO, "ZLIB compression started");

			restartStream();
		} catch (Exception e) {
			printLog(Level.SEVERE, "Can't establish compressed connection", e);
			onError(null, e);
		} finally {
			context.getSessionObject().setProperty(Scope.stream, DISABLE_KEEPALIVE_KEY, Boolean.FALSE);
		}
	}

	protected void setStage(State state) throws JaxmppException {
		if (this.context == null) {
			return;
		}
		State s = this.context.getSessionObject().getProperty(CONNECTOR_STAGE_KEY);
		this.context.getSessionObject().setProperty(Scope.stream, CONNECTOR_STAGE_KEY, state);
		if (s != state) {
			this.context.getSessionObject().setProperty(Scope.stream, CONNECTOR_STAGE_TIMESTAMP_KEY, new Date());
			printLog(Level.FINE, "Connector  state changed: " + s + "->" + state);
			context.getEventBus().fire(new StateChangedEvent(context.getSessionObject(), s, state));
			if (state == State.disconnected) {
				fireOnTerminate(context.getSessionObject());
			}
		}
	}

	private void printLog(Level level, String msg) {
		printLog(level, msg, null);
	}

	private void printLog(Level level, String msg, Throwable e) {
		final StringBuilder sb = new StringBuilder();
		sb.append("[");
		sb.append("scid=").append(oid);
		sb.append(" thread=").append(Thread.currentThread().getId());

		sb.append("] ").append(msg);
		if (e == null) {
			log.log(level, sb.toString());
		} else {
			log.log(level, sb.toString(), e);
		}
	}

	private void closeSocket() {
		printLog(Level.FINE, "Closing socket");
		if (socket.isConnected()) {
			try {
				socket.close();
			} catch (IOException ex) {
				printLog(Level.FINEST, "Problem with closing socket ", ex);
			}
		}
	}

	private X509Certificate[] convertChain(org.bouncycastle.tls.Certificate certificates)
			throws CertificateException, IOException {
		X509Certificate[] result = new X509Certificate[certificates.getLength()];

		for (int i = 0; i < certificates.getLength(); i++) {
			TlsCertificate cert = certificates.getCertificateAt(i);
			java.security.cert.Certificate jsCert = CertificateFactory.getInstance("X.509")
					.generateCertificate(new ByteArrayInputStream(cert.getEncoded()));
			result[i] = (X509Certificate) jsCert;
		}

		return result;
	}

	private Socket createSocket(Entry serverHost) throws IOException {
		Socket socket;
		InetAddress x = InetAddress.getByName(serverHost.getHostname());
		printLog(Level.INFO, "Opening connection to " + x + ":" + serverHost.getPort());

		if (context.getSessionObject().getProperty(Connector.PROXY_HOST) != null) {
			final String proxyHost = context.getSessionObject().getProperty(Connector.PROXY_HOST);
			final int proxyPort = context.getSessionObject().getProperty(Connector.PROXY_PORT);
			Proxy.Type proxyType = context.getSessionObject().getProperty(Connector.PROXY_TYPE);
			if (proxyType == null) {
				proxyType = Proxy.Type.HTTP;
			}

			printLog(Level.INFO, "Using " + proxyType + " proxy: " + proxyHost + ":" + proxyPort);

			SocketAddress addr = new InetSocketAddress(proxyHost, proxyPort);
			Proxy proxy = new Proxy(proxyType, addr);
			socket = new Socket(proxy);
		} else {
			socket = new Socket();
		}

		// if
		// (context.getSessionObject().getProperty(DISABLE_SOCKET_TIMEOUT_KEY)
		// == null
		// || ((Boolean)
		// context.getSessionObject().getProperty(DISABLE_SOCKET_TIMEOUT_KEY)).booleanValue())
		// {
		// socket.setSoTimeout(DEFAULT_SOCKET_TIMEOUT);
		// }
		Integer soTimeout = getTimeout(PLAIN_SOCKET_TIMEOUT_KEY, DEFAULT_SOCKET_TIMEOUT);
		if (soTimeout != null) {
			socket.setSoTimeout(soTimeout);
		}
		socket.setKeepAlive(false);
		socket.setTcpNoDelay(true);
		// writer = new BufferedOutputStream(socket.getOutputStream());
		socket.connect(new InetSocketAddress(x, serverHost.getPort()));

		return socket;
	}

	private Entry getHostFromSessionObject() {
		String serverHost = context.getSessionObject().getProperty(SERVER_HOST);
		Integer port = context.getSessionObject().getProperty(SERVER_PORT);
		if (serverHost == null) {
			return null;
		}
		return new Entry(serverHost, port == null ? 5222 : port);

	}

	private java.security.cert.Certificate getPeerCertificate(SSLSession session) throws SSLPeerUnverifiedException {
		java.security.cert.Certificate[] certificates = session.getPeerCertificates();

		if (certificates == null || certificates.length == 0) {
			return null;
		}
		return certificates[0];
	}

	private void proceedBCTLS() throws JaxmppException {
		printLog(Level.FINE, "Proceeding TLS with Bouncycastle");
		try {
			context.getSessionObject().setProperty(Scope.stream, DISABLE_KEEPALIVE_KEY, Boolean.TRUE);
			final TrustManager[] trustManagers = context.getSessionObject().getProperty(TRUST_MANAGERS_KEY);

			BcTlsCrypto tlsCrypto = new BcTlsCrypto(new SecureRandom());

			final String hostname = getHostname();

			TlsClientProtocol tlsClientProtocol = new TlsClientProtocol(socket.getInputStream(),
																		socket.getOutputStream());

			DefaultTlsClient tlsClient = new DefaultTlsClient(tlsCrypto) {
				public TlsAuthentication getAuthentication() throws IOException {
					return new ServerOnlyTlsAuthentication() {

						@Override
						public void notifyServerCertificate(org.bouncycastle.tls.Certificate certificate)
								throws IOException {
							try {
								final X509Certificate[] certChain = convertChain(certificate);

								TlsAuthentication a = getAuthentication();
								System.out.println(a);
								TlsSession b = getSessionToResume();
								System.out.println(b);
								TlsKeyExchange c = getKeyExchange();
								System.out.println(c);

								Vector z = getSupportedSignatureAlgorithms();
								for (Object o : z) {
									System.out.println("++>" + o + "  " + o.getClass());
								}

								String authType = getAuthType(getKeyExchange());
								System.out.println("AUTH_TYPE=" + authType);

								if (trustManagers != null) {
									for (TrustManager trustManager : trustManagers) {
										if (trustManager instanceof X509TrustManager) {
											((X509TrustManager) trustManager).checkServerTrusted(certChain, authType);
										}
									}
								}

								final Object hnv = SocketConnector.this.context.getSessionObject()
										.getProperty(HOSTNAME_VERIFIER_KEY);

								if (hnv != null && hnv instanceof JaxmppHostnameVerifier) {
									if (!((JaxmppHostnameVerifier) hnv).verify(hostname, certChain[0])) {
										throw new javax.net.ssl.SSLHandshakeException(
												"Cerificate hostname doesn't match domain name you want to connect.");
									}
								} else if (hnv != null && hnv instanceof HostnameVerifier) {
									throw new javax.net.ssl.SSLHandshakeException(
											"javax.net.ssl.HostnameVerifier is not supported! Use tigase.jaxmpp.j2se" +
													".connectors.socket.JaxmppHostnameVerifier instead.");
								}

								SocketConnector.this.context.getSessionObject()
										.setProperty(Scope.stream, TLS_PEER_CERTIFICATE_KEY, certChain[0]);
							} catch (javax.net.ssl.SSLHandshakeException e) {
								e.printStackTrace();
								throw new IOException("Cannot peer validate certificate", e);
							} catch (Exception e) {
								e.printStackTrace();
								throw new RuntimeException(e);
							}

						}

					};
				}

				@Override
				public void notifyHandshakeComplete() throws IOException {
					printLog(Level.INFO, "TLS completed ");
					super.notifyHandshakeComplete();
					byte[] cb = context.exportChannelBinding(ChannelBinding.tls_unique);
					SocketConnector.this.context.getSessionObject().setProperty(Scope.stream, TLS_SESSION_ID_KEY, cb);

					SocketConnector.this.context.getSessionObject()
							.setProperty(Scope.stream, ENCRYPTED_KEY, Boolean.TRUE);
					SocketConnector.this.context.getEventBus()
							.fire(new EncryptionEstablishedHandler.EncryptionEstablishedEvent(
									SocketConnector.this.context.getSessionObject()));

				}
			};

			tlsClientProtocol.connect(tlsClient);

			writer = tlsClientProtocol.getOutputStream();
			reader = new TextStreamReader(tlsClientProtocol.getInputStream());

			restartStream();
		} catch (javax.net.ssl.SSLHandshakeException e) {
			printLog(Level.SEVERE, "Can't establish encrypted connection", e);
			onError(null, e);
		} catch (Exception e) {
			printLog(Level.SEVERE, "Can't establish encrypted connection", e);
			onError(null, e);
		} finally {
			context.getSessionObject().setProperty(Scope.stream, DISABLE_KEEPALIVE_KEY, Boolean.FALSE);
		}
	}

	private void proceedJCETLS() throws JaxmppException {
		printLog(Level.FINE, "Proceeding TLS");
		try {
			context.getSessionObject().setProperty(Scope.stream, DISABLE_KEEPALIVE_KEY, Boolean.TRUE);
			TrustManager[] trustManagers = context.getSessionObject().getProperty(TRUST_MANAGERS_KEY);
			final SSLSocketFactory factory;
			if (trustManagers == null) {
				if (context.getSessionObject().getProperty(SSL_SOCKET_FACTORY_KEY) != null) {
					factory = context.getSessionObject().getProperty(SSL_SOCKET_FACTORY_KEY);
				} else {
					factory = (SSLSocketFactory) SSLSocketFactory.getDefault();
				}
			} else {
				SSLContext ctx = SSLContext.getInstance("TLS");
				final KeyManager[] keyManagers = getKeyManagers();
				ctx.init(keyManagers, trustManagers, new SecureRandom());
				factory = ctx.getSocketFactory();
			}

			SSLSocket s1 = (SSLSocket) factory.createSocket(socket, getHostname(), socket.getPort(), true);

			// if
			// (context.getSessionObject().getProperty(DISABLE_SOCKET_TIMEOUT_KEY)
			// == null
			// || !((Boolean)
			// context.getSessionObject().getProperty(DISABLE_SOCKET_TIMEOUT_KEY)).booleanValue())
			// {
			// s1.setSoTimeout(DEFAULT_SOCKET_TIMEOUT);
			// }
			Integer sslSoTimeout = getTimeout(SSL_SOCKET_TIMEOUT_KEY, 0);
			if (sslSoTimeout != null) {
				s1.setSoTimeout(sslSoTimeout);
			}
			s1.setKeepAlive(false);
			s1.setTcpNoDelay(true);
			s1.setUseClientMode(true);
			s1.addHandshakeCompletedListener(new HandshakeCompletedListener() {

				@Override
				public void handshakeCompleted(HandshakeCompletedEvent arg0) {
					printLog(Level.INFO, "TLS completed " + arg0);
					context.getSessionObject().setProperty(Scope.stream, ENCRYPTED_KEY, Boolean.TRUE);
					context.getEventBus().fire(new EncryptionEstablishedEvent(context.getSessionObject()));
// Removed due to Java API limitations
//					context.getSessionObject().setProperty(Scope.stream, TLS_SESSION_ID_KEY, arg0.getSession().getId());

					try {
						java.security.cert.Certificate[] certs = arg0.getPeerCertificates();
						java.security.cert.Certificate peerCertificate =
								certs == null || certs.length == 0 ? null : certs[0];
						//	context.getSessionObject().setProperty(Scope.stream, TLS_PEER_CERTIFICATE_KEY, peerCertificate);
					} catch (Exception e) {
						printLog(Level.WARNING, "Cannot extract peer certificate", e);
					}
				}
			});
			writer = null;
			reader = null;
			printLog(Level.FINE, "Start handshake");

			final String hostname = getHostname();

			s1.startHandshake();

			final Object hnv = context.getSessionObject().getProperty(HOSTNAME_VERIFIER_KEY);
			if (hnv != null && hnv instanceof HostnameVerifier &&
					!((HostnameVerifier) hnv).verify(hostname, s1.getSession())) {
				throw new javax.net.ssl.SSLHandshakeException(
						"Certificate hostname doesn't match domain name you want to connect.");
			} else if (hnv != null && hnv instanceof JaxmppHostnameVerifier &&
					!((JaxmppHostnameVerifier) hnv).verify(hostname, getPeerCertificate(s1.getSession()))) {
				throw new javax.net.ssl.SSLHandshakeException(
						"Certificate hostname doesn't match domain name you want to connect.");
			}

			socket = s1;
			writer = socket.getOutputStream();
			reader = new TextStreamReader(socket.getInputStream());
			restartStream();
		} catch (javax.net.ssl.SSLHandshakeException e) {
			printLog(Level.SEVERE, "Can't establish encrypted connection", e);
			onError(null, e);
		} catch (Exception e) {
			printLog(Level.SEVERE, "Can't establish encrypted connection", e);
			onError(null, e);
		} finally {
			context.getSessionObject().setProperty(Scope.stream, DISABLE_KEEPALIVE_KEY, Boolean.FALSE);
		}
	}

	private void reconnect(final String newHost) {
		printLog(Level.INFO, "See other host: " + newHost);
		try {
			this.context.getSessionObject().setProperty(RECONNECTING_KEY, Boolean.TRUE);
			terminateAllWorkers();

			this.context.getSessionObject().clear(SessionObject.Scope.stream);
			this.context.getSessionObject().setProperty(SERVER_HOST, newHost);
			worker = null;
			reader = null;
			writer = null;

			this.context.getSessionObject().setProperty(RECONNECTING_KEY, Boolean.TRUE);

			printLog(Level.FINEST, "Waiting for workers termination");

			// start();
		} catch (JaxmppException e) {
			printLog(Level.WARNING, "Error on recconnect", e);
		}
	}

	private void terminateAllWorkers() throws JaxmppException {
		printLog(Level.FINEST, "Terminating all workers ");
		if (this.pingTask != null) {
			this.pingTask.cancel();
			this.pingTask = null;
		}
		// setStage(State.disconnected);
		if (socket != null && socket.isConnected()) {
			synchronized (this) {
				if (closeTimer != null) {
					closeTimer.cancel();
				}
				closeTimer = new Timer("SocketConnectorCloseTimer", true);
				closeTimer.schedule(new TimerTask() {
					@Override
					public void run() {
						try {
							setStage(State.disconnected);
						} catch (JaxmppException e) {
						}
						context = null;
						closeSocket();
						synchronized (SocketConnector.this) {
							if (closeTimer != null) {
								closeTimer.cancel();
								closeTimer = null;
							}
						}
					}
				}, 3 * 1000);
			}
		} else {
			try {
				setStage(State.disconnected);
			} catch (JaxmppException e) {
			}
		}

		// is there a need for this?
		try {
			if (worker != null) {
				worker.interrupt();
			}
		} catch (Exception e) {
			printLog(Level.FINEST, "Problem with interrupting w2", e);
		}
		try {
			if (timer != null) {
				timer.cancel();
			}
		} catch (Exception e) {
			printLog(Level.FINEST, "Problem with canceling timer", e);
		} finally {
			timer = null;
		}
	}

	private void terminateStream() throws JaxmppException {
		final State state = getState();
		if (state == State.connected || state == State.connecting || state == State.disconnecting) {
			String x = "</stream:stream>";
			printLog(Level.FINE, "Terminating XMPP Stream");
			send(x.getBytes(UTF_CHARSET));
		} else {
			printLog(Level.FINE, "Stream terminate not sent, because of connection state==" + state);
		}
	}

	private void workerTerminated(final Worker worker) {
		try {
			synchronized (this) {
				if (closeTimer != null) {
					closeTimer.cancel();
					closeTimer = null;
				}
				setStage(State.disconnected);
			}
		} catch (JaxmppException e) {
		}
		printLog(Level.FINEST, "Worker terminated");
		try {
			if (this.context.getSessionObject().getProperty(RECONNECTING_KEY) == Boolean.TRUE) {
				this.context.getSessionObject().setProperty(RECONNECTING_KEY, null);
				context.getEventBus().fire(new HostChangedEvent(context.getSessionObject()));
				printLog(Level.FINEST, "Restarting...");
				start();
			} else {
				context.getEventBus().fire(new DisconnectedHandler.DisconnectedEvent(context.getSessionObject()));
			}
		} catch (Exception e) {
			printLog(Level.WARNING, "Problem : " + e.getMessage());
		}
	}

	public interface DnsResolver {

		List<Entry> resolve(String hostname);
	}

	/**
	 * see-other-host
	 */
	public interface HostChangedHandler
			extends EventHandler {

		void onHostChanged(SessionObject sessionObject);

		class HostChangedEvent
				extends JaxmppEvent<HostChangedHandler> {

			public HostChangedEvent(SessionObject sessionObject) {
				super(sessionObject);
			}

			@Override
			public void dispatch(HostChangedHandler handler) {
				handler.onHostChanged(sessionObject);
			}

		}
	}

	public final static class Entry {

		private final String hostname;

		private final Integer port;

		public Entry(String host, Integer port) {
			this.hostname = host;
			this.port = port;
		}

		public String getHostname() {
			return hostname;
		}

		public Integer getPort() {
			return port;
		}

		@Override
		public String toString() {
			return hostname + ":" + port;
		}

	}

}
