/*
 * Tigase XMPP Client Library
 * Copyright (C) 2006-2012 "Bartosz Ma��kowski" <bartosz.malkowski@tigase.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. Look for COPYING file in the top folder.
 * If not, see http://www.gnu.org/licenses/.
 */
package tigase.jaxmpp.core.client.xmpp.modules.roster;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import tigase.jaxmpp.core.client.AsyncCallback;
import tigase.jaxmpp.core.client.BareJID;
import tigase.jaxmpp.core.client.Property;
import tigase.jaxmpp.core.client.SessionObject;
import tigase.jaxmpp.core.client.exceptions.JaxmppException;
import tigase.jaxmpp.core.client.xml.XMLException;

/**
 * Storage for keeping roster.
 */
public abstract class RosterStore implements Property {

	static interface Handler {

		void add(BareJID jid, String name, Collection<String> groups, AsyncCallback asyncCallback) throws XMLException,
				JaxmppException;

		void cleared();

		void remove(BareJID jid) throws XMLException, JaxmppException;

		void update(RosterItem item) throws XMLException, JaxmppException;
	}

	public static interface Predicate {
		boolean match(RosterItem item);
	}

	private Handler handler;
	protected SessionObject sessionObject;

	/**
	 * Adds new contact to roster.
	 * 
	 * @param jid
	 *            JID of buddy
	 * @param name
	 *            name of buddy
	 * @param asyncCallback
	 *            callback
	 */
	public void add(BareJID jid, String name, AsyncCallback asyncCallback) throws XMLException, JaxmppException {
		add(jid, name, new ArrayList<String>(), asyncCallback);
	}

	/**
	 * Adds new contact to roster.
	 * 
	 * @param jid
	 *            JID of buddy
	 * @param name
	 *            name of buddy
	 * @param groups
	 *            collection of groups name
	 * @param asyncCallback
	 *            callback
	 */
	public void add(BareJID jid, String name, Collection<String> groups, AsyncCallback asyncCallback) throws XMLException,
			JaxmppException {
		if (this.handler != null)
			this.handler.add(jid, name, groups, asyncCallback);
	}

	/**
	 * Adds new contact to roster.
	 * 
	 * @param jid
	 *            JID of buddy
	 * @param name
	 *            name of buddy
	 * @param groups
	 *            array of groups name
	 * @param asyncCallback
	 *            callback
	 */
	public void add(BareJID jid, String name, String[] groups, AsyncCallback asyncCallback) throws XMLException,
			JaxmppException {
		ArrayList<String> x = new ArrayList<String>();
		if (groups != null)
			for (String string : groups) {
				x.add(string);
			}
		add(jid, name, x, asyncCallback);
	}

	protected abstract Set<String> addItem(RosterItem item);

	protected abstract Set<String> calculateModifiedGroups(final HashSet<String> groupsOld);
	
	/**
	 * Clears storage.
	 */
	public void clear() {
		removeAll();
		if (this.handler != null)
			handler.cleared();
	}

	/**
	 * Returns {@linkplain RosterItem} of given bare JID.
	 * 
	 * @param jid
	 *            bare JID.
	 * @return roster item.
	 */
	public abstract RosterItem get(BareJID jid);

	/**
	 * Returns all buddies from roster.
	 * 
	 * @return all roster items.
	 */
	public List<RosterItem> getAll() {
		return getAll(null);
	}

	/**
	 * Returns all roster items selected by selector.
	 * 
	 * @param predicate
	 *            selector.
	 * @return all matched roster items.
	 */
	public abstract List<RosterItem> getAll(final Predicate predicate);

	/**
	 * Returns number of roster items in storage.
	 * 
	 * @return number of roster items in storage.
	 */
	public abstract int getCount();

	/**
	 * Get all known groups of buddies.
	 * 
	 * @return collection of group names.
	 */
	public abstract Collection<? extends String> getGroups();

	@Override
	public Class<RosterStore> getPropertyClass() {
		return RosterStore.class;
	}

	/**
	 * Removes buddy from roster.
	 * 
	 * @param jid
	 *            jid of buddy to remove.
	 */
	public void remove(BareJID jid) throws JaxmppException {
		if (handler != null)
			this.handler.remove(jid);
	}

	public abstract void removeAll();

	protected abstract void removeItem(BareJID jid);

	void setHandler(Handler handler) {
		this.handler = handler;
	}

	/**
	 * Sends changed RosterItem to server.
	 * 
	 * @param item
	 *            changed roster item.
	 */
	public void update(RosterItem item) throws JaxmppException {
		if (this.handler != null)
			this.handler.update(item);

	}

	public void setSessionObject(SessionObject sessionObject) {
		this.sessionObject = sessionObject;
	}
}